package com.topologi.schematron;

/*
 * EmbRNGValidator.java - Perform validation against a RELAX-NG schema and
 * Schematron validation for any embedded Schematron rules.
 * Copyright (C) 2002 Eddie Robertsson
 *
 * You may use and modify this package for any purpose. Redistribution is
 * permitted, in both source and binary form, provided that this notice
 * remains intact in all source distributions of this package.
 */

import java.io.File;
import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import com.thaiopensource.relaxng.util.Sax2XMLReaderCreator;
import com.thaiopensource.relaxng.util.ValidationEngine;

/**
 * <p>A simple java implementation of a validator that handles RELAX-NG validation
 * as well as validation against Schematron rules embedded in the RELAX-NG
 * schema.</p>
 * 
 * <p>Schematron validation is performed using the <code>SchtrnValidator</code> 
 * class and RELAX-NG validation is performed using James Clark's <code>Jing</code>.</p>
 * 
 * <p>The implementation can be used both as an API (by using the <code>validate(...)</code>
 * methods) and from the commandline (the <code>main(...)</code> method).</p>
 * 
 * <p>It should be noted that this is a simple implementation of a Schematron
 * validator. No effort has been made to optimize the code in terms of 
 * performance or memory usage.</p>
 *
 * @see: com.topologi.schematron.SchtrnValidator
 * @see: http://www.thaiopensource.com/relaxng/jing.html
 * @author: Eddie Robertsson
 */
public class EmbRNGValidator {

	/**
	* The Schematron validator that is used to validate the embedded
	* Schematron rules
	*/
	private SchtrnValidator schtrnValidator = null;
	/**
	* The <code>TransformerFactory</code> used to the get the 
	* <code>Transformer</code> objects.
	*/
	private TransformerFactory theFactory;
	/**
	* A monitor that waits for the results from both RELAX-NG validation and
	* embedded Schematron validation before continuing
	*/
	private ValidationMonitor valMonitor;
	/**
	* The XSLT stylesheet (as a StreamSource) that is used to extract the 
	* embedded Schematron rules from the RELAX-NG schema
	*/
	private StreamSource extractorStylesheet = null;
	/**
	* Determine which option that should be used for validation
	*/
	private short validationOption = EmbRNGValidator.BOTH;

	/**
	* Validate against both RELAX-NG and embedded Schematron
	*/
	public static final short BOTH = 0;
	/**
	* Validate against RELAX-NG only, disregarding any embedded Schematron
	*/
	public static final short ONLY_RNG = 1;
	/**
	* Validate against embedded Schematron only
	*/
	public static final short ONLY_SCHTRN = 2;

	//------------ Inner classes ------------//

	/**
	 * Monitor that locks the validation thread until both the results from
	 * Schematron validation and the results from RELAX-NG validation
	 * has been provided.
	 * Creation date: (5/3/2002 12:01:52 PM)
	 * @author: Eddie Robertsson
	 */
	private class ValidationMonitor {
		/**
		* A boolean used to wait for RELAX-NG validation.
		*/
		private boolean rngIsDone;
		/**
		* A boolean used to wait for Schematron validation.
		*/
		private boolean schtrnIsDone;
		/**
		 * Create a ValidationMonitor object
		 */
		public ValidationMonitor() {
			rngIsDone = false;
			schtrnIsDone = false;
		}
		/**
		* Waits for both RELAX-NG and Schematron validation
		*/
		public synchronized boolean waitForValidation() {
			try {
				while (!rngIsDone || !schtrnIsDone) {
					wait();
				}
			} catch (InterruptedException e) {
				return false;
			}
			return true;
		}
		/**
		* Called when Schematron validation has finished
		*/
		public synchronized void schtrnDone() {
			schtrnIsDone = true;
			notifyAll();
		}
		/**
		* Called when RELAX-NG validation has finished
		*/
		public synchronized void rngDone() {
			rngIsDone = true;
			notifyAll();
		}
	}

	//-------------- Methods --------------//

	/**
	 * Create an EmbRNGValidator object.
	 */
	public EmbRNGValidator() {
		theFactory = TransformerFactory.newInstance();
		schtrnValidator = new SchtrnValidator();
	}
	/**
	 * Run RELAX-NG validation from the commandline.
	 * 
	 * <pre>
	 * Usage: EmbRNGValidator 'xml' 'schema' {param=value}..." 
	 * 
	 * 'xml'       : The location of the XML instance document
	 * 'schema'    : The location of the RELAX-NG schema
	 * 
	 * Supported parameters:
	 *    diagnose  : 'yes'|'no'  If yes then diagnostics will be
	 *                 included in the result if specified in the Schematron
	 *                 schema and supported by the engine stylesheet.
	 *    phase     :  The name of the phase that should be used for validation.
	 *                 If unspecified then all phases will be used unless
	 *                 the schema specifies the 'defaultPhase' attribute.
	 * </pre>
	 * 
	 * @param: args The arguments sent to the application
	 */
	public static void main(String[] args) {

		// There must be at least two arguments
		if (args.length < 2) {
			System.out.println("Not enough arguments.");
			printUsage();
			System.exit(0);
		}
		// The xml instance is the first argument
		String xml = args[0];
		// The RELAX-NG schema is the second argument
		String schema = args[1];
		// Create the RELAX-NG validator
		EmbRNGValidator validator = new EmbRNGValidator();
		// Get any parameters that should be used for Schematron validation
		validator.getSchtrnValidator().setParams(SchtrnValidator.parseArgs(args));
		// Set the validation option to get results from both RELAX-NG validation
		// and embedded Schematron validation
		validator.setValidationOption(EmbRNGValidator.BOTH);
		// Set the extractor stylesheet to use
		validator.setExtractorStylesheet("Scripts" + File.separator + "RNG2Schtrn.xsl");
		// Print the result to System.out
		System.out.println(validator.validate(xml, schema));
	}
	/**
	 * Print a message to System.out with the options for running validation
	 * from the commandline
	 */
	private static void printUsage() {
		System.out.println(
			"Usage: EmbRNGValidator 'xml' 'schema' {param=value}...\n\n"
				+ "<xml>       : The location of the XML instance document\n"
				+ "<schema>    : The location of the RELAX-NG schema\n\n"
				+ "Supported parameters:\n"
				+ "  diagnose  : <yes>|<no>  If yes then diagnostics will be \n"
				+ "               included in the result if specified in the Schematron\n"
				+ "               schema and supported by the engine stylesheet.\n"
				+ "  phase     :  The name of the phase that should be used for validation.\n"
				+ "               If unspecified then all phases will be used unless\n"
				+ "               the schema specifies the 'defaultPhase' attribute.");
	}
	/**
	 * <p>Validate an XML instance document against a RELAX-NG schema that may
	 * have embedded Schematron rules.</p>
	 * <p>The method takes the paths to the XML instance document and RELAX-NG 
	 * schema as parameters.</p>
	 * <p>If the paths starts with 'file:' it is assumed that the location is
	 * a valid URL and it will be used as such. Otherwise the path provided will
	 * be converted to a URL before validation</p>
	 * @param: xml The path to the XML instance document
	 * @param: schema The path to the RELAX-NG schema
	 * @return: The results from RELAX-NG validation merged with the results
	 * from embedded Schematron validation as a <code>String</code>
	 */
	public String validate(String xml, String schema) {
		String xmlURL = SchtrnValidator.system_To_URL(xml);
		if (xmlURL == null) {
			System.out.println("The xml file location is not valid: " + xml);
			System.exit(0);
		}
		String schemaURL = SchtrnValidator.system_To_URL(schema);
		if (schemaURL == null) {
			System.out.println("The RELAX-NG schema location is not valid: " + schema);
			System.exit(0);
		}
		return validate(new StreamSource(xmlURL), new StreamSource(schemaURL));
	}
	/**
	 * <p>Validate an XML instance document against a RELAX-NG schema that may
	 * have embedded Schematron rules.</p>
	 * <p>The method takes a <code>StreamSource</code> representation of the
	 * XML instance document and RELAX-NG schema as parameters.</p>
	 * @param: xml The XML instance document provided as a 
	 * <code>StreamSource</code>
	 * @param: schema The Schematron schema provided as a 
	 * <code>StreamSource</code>
	 * @return: The results from RELAX-NG validation merged with the results
	 * from embedded Schematron validation as a <code>String</code>
	 */
	public String validate(StreamSource xml, StreamSource schema) {
		StringBuffer results = new StringBuffer();
		// Create the monitor that waits for the result
		valMonitor = new ValidationMonitor();
		// Check which validation option is selected
		if (validationOption == ONLY_RNG) {
			// We're not interested in Schematorn results so report Schematron
			// validation as finished on the monitor
			valMonitor.schtrnDone();
		} else if (validationOption == ONLY_SCHTRN) {
			// We're not interested in RELAX-NG results so report RELAX-NG
			// validation as finished on the monitor
			valMonitor.rngDone();
		}
		// Perform the validation
		if (validationOption == ONLY_RNG) {
			// Only RELAX-NG validation
			performRNGValidation(xml, schema, results);
		} else if (validationOption == ONLY_SCHTRN) {
			// Only Schematron validation
			performSchtrnValidation(xml, schema, results);
		} else {
			// Both
			performSchtrnValidation(xml, schema, results);
			performRNGValidation(xml, schema, results);
		}
		// Wait for validation to finish
		if (!valMonitor.waitForValidation()) {
			// An error occured
			results.append("\n\nAn error occurred when waiting for the validation to finish.");
		}
		return results.toString();
	}
	/**
	 * <p>Perform XMl validation against the RELAX-NG schema in a separate thread.</p>
	 * <p>The results will be appended to the provided <code>StringBuffer</code>
	 * that is synchronized to make it thread safe.</p>
	 * <p>The method takes a <code>StreamSource</code> representation of the
	 * XML instance document and RELAX-NG schema as parameters as well as the
	 * <code>StringBuffer</code> that the validation results are appended to.</p>
	 * @param: xml The XML instance document provided as a 
	 * <code>StreamSource</code>
	 * @param: schema The Schematron schema provided as a 
	 * <code>StreamSource</code>
	 * @param: results The StringBuffer which the results are appended to
	 */
	private void performRNGValidation(final StreamSource xml, final StreamSource schema, final StringBuffer results) {
		Thread rngValidation = new Thread() {
			public void run() {
				String rngResults;
				// Do RELAX-NG validation
				rngResults = executeRNGValidation(xml, schema);
				// Append the results
				synchronized (results) {
					results.append("Validation results from RELAX-NG:\n\n");
					results.append(rngResults);
					results.append("\n\n\n");
				}
				// Let the monitor know that RELAX-NG validation is finished
				valMonitor.rngDone();
			}
		};
		rngValidation.start();
	}
	/**
	 * <p>Execute RELAX-NG validation of the XML document against the RELAX-NG
	 * schema. This implementation use Jing to validate the XML and Jing uses
	 * the XMLReader implementation from Saxon so that no more external jar
	 * files are needed.</p>
	 * <p>The XML document and RELAX-NG schema are passed as parameters in the
	 * form of <code>StreamSource</code> objects and the result is returned
	 * as a <code>String</code>.</p>
	 * @param: xml The XML instance document as a <code>StreamSource</code>
	 * @param: schema The RELAX-NG schema as a <code>StreamSource</code>
	 * @return: String The validation results
	 */
	private String executeRNGValidation(StreamSource xml, StreamSource schema) {
		// StringBuffer with the results
		StringBuffer results = new StringBuffer();
		// Create the validation engine
		// Use the XMLReader in Saxon (aelfred) so that we don't have to
		// include any more jar files
		// More detailed errors can be caught by implementing an ErrorHandler
		// and pass that to the ValidationEngine if needed.
		ValidationEngine engine = new ValidationEngine(new Sax2XMLReaderCreator("com.icl.saxon.aelfred.SAXDriver"), null, true);
		try {
			// Load the schema in the ValidationEngine
			InputSource schemaSource = new InputSource(schema.getInputStream());
			schemaSource.setSystemId(schema.getSystemId());
			engine.loadSchema(schemaSource);
			// Validate the XML against the schema
			InputSource xmlSource = new InputSource(xml.getInputStream());
			xmlSource.setSystemId(xml.getSystemId());
			engine.validate(xmlSource);
		} catch (SAXParseException se) {
			// Validation error
			results.append(se.getSystemId()).append(":").append(se.getLineNumber());
			results.append(":").append(se.getColumnNumber()).append(":");
			results.append(se.getMessage()).append("\n");
		} catch (SAXException e) {
			// Something went wrong
			results.append(e.getMessage()).append("\n");
		} catch (Exception e) {
			// Something went wrong
			results.append(e.getMessage()).append("\n");
		}
		if (results.length() == 0) {
			// No validation errors
			results.append("RELAX-NG validation passed without errors.").append("\n");
		}
		return results.toString();
	}
	/**
	 * <p>Perform XML validation against the embedded Schematron rules in a 
	 * separate thread.</p>
	 * <p>The results will be appended to the provided <code>StringBuffer</code>
	 * that is synchronized to make it thread safe.</p>
	 * <p>The method takes a <code>StreamSource</code> representation of the
	 * XML instance document and RELAX-NG schema as parameters as well as the
	 * <code>StringBuffer</code> that the validation results are appended to.</p>
	 * @param: xml The XML instance document provided as a 
	 * <code>StreamSource</code>
	 * @param: schema The Schematron schema provided as a 
	 * <code>StreamSource</code>
	 * @param: results The StringBuffer which the results are appended to
	 */
	private void performSchtrnValidation(final StreamSource xml, final StreamSource schema, final StringBuffer results) {
		Thread schtrnValidation = new Thread() {
			public void run() {
				String schtrnResults;
				try {
					// Extract the embedded Schematron rules
					StreamSource schtrnSchema = extractEmbeddedRules(schema);
					// Set the system id to be the same as the RELAX-NG schema
					// NOTE: This is important for the transformations to work
					// otherwise an exception will occurr for Schematron
					// validation
					schtrnSchema.setSystemId(schema.getSystemId());
					// Set the engine stylesheet to use for Schematron validation
					schtrnValidator.setEngineStylesheet("Scripts" + File.separator + "schematron-diagnose.xsl");
					// Perform Schematron validation
					schtrnResults = schtrnValidator.validate(xml, schtrnSchema);
				} catch (TransformerConfigurationException tce) {
					schtrnResults = tce.getMessageAndLocation();
				} catch (TransformerException te) {
					schtrnResults = te.getMessageAndLocation();
				} catch (Exception e) {
					schtrnResults = e.getMessage();
				}
				// Append the results
				synchronized (results) {
					results.append("Validation results from embedded Schematron rules:");
					results.append(schtrnResults);
					results.append("\n\n\n");
				}
				// Let the monitor know that Schematron validation is finished
				valMonitor.schtrnDone();
			}
		};
		schtrnValidation.start();
	}
	/**
	 * <p>Extract the embedded Schematron rules from the RELAX-NG schema provided
	 * as a <code>StreamSource</code>.</p>
	 * <p>The returned result is a <code>StreamSource</code> of the Schematron
	 * schema created by the extraction of the rules.</p>
	 * <p>Improvements: To improve efficiency code can be added to first
	 * check that the RELAX-NG schema actually contains embedded Schematron rules
	 * by checking for any element in the Schematron namespace.</p>
	 * @param: schema The RELAX-NG schema with embedded rules
	 * @return: StreamSource The concatenated Schematron schema
	 */
	private StreamSource extractEmbeddedRules(StreamSource schema) throws TransformerConfigurationException, TransformerException {
		// Set the stylesheet to use for the extraction
		Transformer transformer = theFactory.newTransformer(extractorStylesheet);
		StringWriter result = new StringWriter();
		// Extract the Schematron rules
		transformer.transform(schema, new StreamResult(result));
		return new StreamSource(new StringReader(result.toString()));
	}
	/**
	 * Returns the extractor stylesheet.
	 * @return StreamSource
	 */
	public StreamSource getExtractorStylesheet() {
		return extractorStylesheet;
	}
	/**
	 * Sets the extractor stylesheet by providing a path to the location of the
	 * stylesheet file.
	 * @param extractorFile The path to the stylesheet
	 */
	public void setExtractorStylesheet(String extractorFile) {
		if (extractorFile != null) {
			extractorStylesheet = new StreamSource(SchtrnValidator.system_To_URL(extractorFile));
		}
	}
	/**
	 * Returns the validation option.
	 * @return short
	 */
	public short getValidationOption() {
		return validationOption;
	}
	/**
	 * Sets the validation option.
	 * @param validationOption The validationOption to set
	 */
	public void setValidationOption(short validationOption) {
		this.validationOption = validationOption;
	}
	/**
	 * Returns the Schematron validator.
	 * @return SchtrnValidator
	 */
	public SchtrnValidator getSchtrnValidator() {
		return schtrnValidator;
	}

	/**
	 * Sets the Schematron validator.
	 * @param schtrnValidator The schtrnValidator to set
	 */
	public void setSchtrnValidator(SchtrnValidator schtrnValidator) {
		this.schtrnValidator = schtrnValidator;
	}
}
